<?php
/**
 * Plugin Name: UPT Live Images (Staging Path-Aware)
 * Description: Importiert Bilder von einer Live-Base-URL (Domain/Pfad) beim Speichern lokal in die Mediathek (Staging) und ersetzt URLs im Content.
 * Version:     1.1.1
 * Author:      Jola & ChatGPT
 */

if ( ! defined('ABSPATH') ) exit;

final class UPT_Live_Images_Staging_PathAware {
	const OPT_KEY          = 'upt_live_images_pathaware_settings';
	const META_ORIGIN_URL  = '_upt_origin_url';
	private static $lock   = false;

	public function __construct() {
		add_action('admin_menu', [ $this, 'admin_menu' ]);
		add_action('admin_init', [ $this, 'register_settings' ]);
		add_action('save_post',  [ $this, 'on_save_post' ], 10, 3);
	}

	private function defaults() : array {
		return [
			'enabled'          => 1,
			'live_base_url'    => 'https://meineLiveSite.de/',
			'staging_base_url' => 'https://StagingSite.de/',
		];
	}

	private function get_settings() : array {
		$opt = get_option(self::OPT_KEY, []);
		if ( ! is_array($opt) ) $opt = [];
		return wp_parse_args($opt, $this->defaults());
	}

	/**
	 * Normalisiert Base-URL zu: host + path (mit trailing slash)
	 * Muss public sein, ist aber nur intern genutzt – trotzdem ok.
	 */
	public function normalize_base(string $url) : array {
		$url = trim($url);
		if ($url === '') return ['host'=>'','path'=>'/'];

		// wenn scheme fehlt: https annehmen (nur zum Parsen)
		if ( ! preg_match('~^https?://~i', $url) ) $url = 'https://' . $url;

		$p = wp_parse_url($url);
		$host = isset($p['host']) ? strtolower($p['host']) : '';
		$host = preg_replace('~^www\.~i', '', $host);

		$path = isset($p['path']) ? $p['path'] : '/';
		if ($path === '') $path = '/';
		if ($path[0] !== '/') $path = '/' . $path;
		if (substr($path, -1) !== '/') $path .= '/';

		return ['host'=>$host, 'path'=>$path];
	}

	/**
	 * WICHTIG: sanitize_callback wird von WP von außen aufgerufen -> muss public sein.
	 */
	public function sanitize_settings($value) : array {
		$value = is_array($value) ? $value : [];
		$enabled = ! empty($value['enabled']) ? 1 : 0;

		$live = is_string($value['live_base_url'] ?? '') ? trim($value['live_base_url']) : '';
		$stag = is_string($value['staging_base_url'] ?? '') ? trim($value['staging_base_url']) : '';

		$live = preg_replace('~\s+~', '', $live);
		$stag = preg_replace('~\s+~', '', $stag);

		return [
			'enabled'          => $enabled,
			'live_base_url'    => $live,
			'staging_base_url' => $stag,
		];
	}

	public function register_settings() : void {
		register_setting(
			'upt_live_images_pathaware_group',
			self::OPT_KEY,
			[
				'type'              => 'array',
				'sanitize_callback' => [ $this, 'sanitize_settings' ],
				'default'           => $this->defaults(),
			]
		);
	}

	public function admin_menu() : void {
		add_options_page(
			'UPT Live Images',
			'UPT Live Images',
			'manage_options',
			'upt-live-images',
			[ $this, 'render_settings_page' ]
		);
	}

	public function render_settings_page() : void {
		if ( ! current_user_can('manage_options') ) wp_die('Keine Berechtigung.');

		$set = $this->get_settings();
		?>
		<div class="wrap">
			<h1>UPT Live Images (Staging)</h1>
			<p><strong>Ziel:</strong> Live-Bilder anzeigen (Fallback) und beim Speichern in Staging lokal importieren.</p>

			<form method="post" action="options.php">
				<?php settings_fields('upt_live_images_pathaware_group'); ?>

				<table class="form-table" role="presentation">
					<tr>
						<th scope="row">Aktiv</th>
						<td>
							<label>
								<input type="checkbox"
									name="<?php echo esc_attr(self::OPT_KEY); ?>[enabled]"
									value="1" <?php checked(!empty($set['enabled'])); ?>>
								Beim Speichern automatisch importieren
							</label>
						</td>
					</tr>

					<tr>
						<th scope="row"><label for="upt_live_base">Live Base-URL (Quelle)</label></th>
						<td>
							<input id="upt_live_base" type="text" class="regular-text"
								name="<?php echo esc_attr(self::OPT_KEY); ?>[live_base_url]"
								value="<?php echo esc_attr($set['live_base_url']); ?>"
								placeholder="https://meineLiveSite.de/">
							<p class="description">Mit oder ohne https://. Pfad optional. Beispiel: <code>https://meineLiveSite.de/</code></p>
						</td>
					</tr>

					<tr>
						<th scope="row"><label for="upt_staging_base">Staging Base-URL (Ziel / Sicherheitscheck)</label></th>
						<td>
							<input id="upt_staging_base" type="text" class="regular-text"
								name="<?php echo esc_attr(self::OPT_KEY); ?>[staging_base_url]"
								value="<?php echo esc_attr($set['staging_base_url']); ?>"
								placeholder="https://StagingSite.de/">
							<p class="description">Hier darf ein Pfad stehen, z.B. <code>StagingSite.de</code>. Plugin läuft nur, wenn die aktuelle Site damit übereinstimmt.</p>
						</td>
					</tr>
				</table>

				<?php submit_button('Speichern'); ?>
			</form>
		</div>
		<?php
	}

	private function current_base() : array {
		return $this->normalize_base( home_url('/') );
	}

	private function env_ok(array $set) : bool {
		$need = $this->normalize_base( (string)($set['staging_base_url'] ?? '') );
		if ($need['host'] === '') return true;

		$cur = $this->current_base();

		if ($cur['host'] !== $need['host']) return false;

		return (strpos($cur['path'], $need['path']) === 0) || ($cur['path'] === $need['path']);
	}

	private function is_live_url(string $url, array $live) : bool {
		$p = wp_parse_url($url);
		if ( empty($p['host']) ) return false;

		$host = strtolower((string)$p['host']);
		$host = preg_replace('~^www\.~i', '', $host);

		if ($host !== $live['host']) return false;

		$path = (string)($p['path'] ?? '');
		if ($path === '') $path = '/';

		if ( $live['path'] !== '/' && strpos($path, $live['path']) !== 0 ) return false;

		return (stripos($path, '/wp-content/uploads/') !== false);
	}

	public function on_save_post($post_id, $post, $update) : void {
		if ( self::$lock ) return;
		if ( wp_is_post_revision($post_id) || wp_is_post_autosave($post_id) ) return;
		if ( ! is_admin() ) return;
		if ( ! $post || empty($post->post_content) ) return;
		if ( ! current_user_can('edit_post', $post_id) ) return;

		$set = $this->get_settings();
		if ( empty($set['enabled']) ) return;
		if ( ! $this->env_ok($set) ) return;

		$live = $this->normalize_base( (string)$set['live_base_url'] );
		if ( $live['host'] === '' ) return;

		$content = (string)$post->post_content;

		$urls = [];

		if ( preg_match_all('~<img[^>]+src=["\']([^"\']+)["\']~i', $content, $m1) ) {
			$urls = array_merge($urls, $m1[1]);
		}
		if ( preg_match_all('~srcset=["\']([^"\']+)["\']~i', $content, $m2) ) {
			foreach ($m2[1] as $srcset) {
				foreach ( preg_split('~\s*,\s*~', $srcset) as $part ) {
					$part = trim($part);
					if ($part === '') continue;
					$u = preg_split('~\s+~', $part)[0] ?? '';
					if ($u) $urls[] = $u;
				}
			}
		}

		$urls = array_values(array_unique(array_filter(array_map('trim', $urls))));
		if ( ! $urls ) return;

		require_once ABSPATH . 'wp-admin/includes/media.php';
		require_once ABSPATH . 'wp-admin/includes/file.php';
		require_once ABSPATH . 'wp-admin/includes/image.php';

		$repl = [];

		foreach ($urls as $url) {
			if ( ! $this->is_live_url($url, $live) ) continue;

			$existing = $this->find_attachment_by_origin_url($url);
			if ($existing) {
				$new = wp_get_attachment_url($existing);
				if ($new) $repl[$url] = $new;
				continue;
			}

			$att_id = media_sideload_image($url, $post_id, null, 'id');
			if ( is_wp_error($att_id) || ! $att_id ) continue;

			update_post_meta($att_id, self::META_ORIGIN_URL, esc_url_raw($url));

			$new = wp_get_attachment_url($att_id);
			if ($new) $repl[$url] = $new;
		}

		if ( ! $repl ) return;

		$new_content = strtr($content, $repl);

		self::$lock = true;
		wp_update_post([
			'ID'           => $post_id,
			'post_content' => $new_content,
		]);
		self::$lock = false;
	}

	private function find_attachment_by_origin_url(string $url) : int {
		$q = new WP_Query([
			'post_type'      => 'attachment',
			'post_status'    => 'inherit',
			'fields'         => 'ids',
			'posts_per_page' => 1,
			'meta_query'     => [
				[
					'key'   => self::META_ORIGIN_URL,
					'value' => $url,
				]
			],
			'no_found_rows'  => true,
		]);

		return ! empty($q->posts[0]) ? (int)$q->posts[0] : 0;
	}
}

new UPT_Live_Images_Staging_PathAware();
